"""
Configuration management for Crypto AI Trader
"""
from pydantic_settings import BaseSettings
from typing import List


class Settings(BaseSettings):
    """Application settings loaded from environment variables"""
    
    # AI Provider
    anthropic_api_key: str
    ai_model: str = "claude-sonnet-4-5-20250929"
    ai_check_interval_seconds: int = 60
    ai_min_confidence: float = 0.70
    ai_max_trades_per_day: int = 20
    
    # Database
    database_url: str
    
    # Trading Settings
    paper_trading: bool = True
    initial_balance_usdt: float = 10000.0
    max_position_size_pct: float = 0.30
    risk_per_trade_pct: float = 0.02
    max_open_positions: int = 5
    
    # Exchange API Keys (optional)
    binance_api_key: str = ""
    binance_secret: str = ""
    coinbase_api_key: str = ""
    coinbase_secret: str = ""
    kraken_api_key: str = ""
    kraken_secret: str = ""
    
    # Exchanges & Symbols
    active_exchanges: str = "binance,coinbase"
    default_symbols: str = "BTC/USDT,ETH/USDT"
    default_timeframe: str = "5m"
    
    # Simulation Settings
    maker_fee: float = 0.001
    taker_fee: float = 0.001
    slippage_pct: float = 0.0005
    
    # Backend Settings
    backend_host: str = "0.0.0.0"
    backend_port: int = 8000
    log_level: str = "INFO"
    
    @property
    def exchanges_list(self) -> List[str]:
        """Get list of active exchanges"""
        return [e.strip() for e in self.active_exchanges.split(",")]
    
    @property
    def symbols_list(self) -> List[str]:
        """Get list of default symbols"""
        return [s.strip() for s in self.default_symbols.split(",")]
    
    class Config:
        env_file = ".env"
        case_sensitive = False


# Global settings instance
settings = Settings()
