"""
Main FastAPI Application
"""
from fastapi import FastAPI, WebSocket
from fastapi.middleware.cors import CORSMiddleware
from contextlib import asynccontextmanager
import asyncio
import logging
from datetime import datetime

from app.config import settings
from app.database import engine, Base
from app.api.routes import router
from app.api.websocket import websocket_endpoint, manager
from app.services.market_data import market_data_service
from app.services.ai_strategy import ai_strategy_service
from app.services.portfolio import portfolio_service
from app.services.execution import execution_simulator
from app.database import get_db_context
from app.utils.indicators import calculate_indicators
from app.models import AISignal

# Configure logging
logging.basicConfig(
    level=getattr(logging, settings.log_level),
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Background task reference
background_task = None


async def ai_trading_loop():
    """
    Main AI trading loop - runs every N seconds
    """
    logger.info("AI trading loop started")
    
    while True:
        try:
            await asyncio.sleep(settings.ai_check_interval_seconds)
            
            logger.info("Running AI analysis cycle...")
            
            # Analyze each symbol
            for symbol in settings.symbols_list:
                try:
                    # Get market data
                    candles = await market_data_service.fetch_ohlcv(
                        'binance', symbol, '5m', limit=100
                    )
                    
                    if not candles or len(candles) < 20:
                        logger.warning(f"Insufficient candle data for {symbol}")
                        continue
                    
                    # Calculate indicators
                    indicators = calculate_indicators(candles)
                    
                    # Get current price
                    ticker = await market_data_service.fetch_ticker('binance', symbol)
                    if not ticker:
                        continue
                    
                    current_price = ticker['last']
                    
                    # Get portfolio state
                    with get_db_context() as db:
                        current_prices = {symbol: current_price}
                        portfolio = portfolio_service.get_state(db, current_prices)
                        
                        # Prepare market data for AI
                        market_data = {
                            'current_price': current_price,
                            'bid': ticker['bid'],
                            'ask': ticker['ask'],
                            'volume_24h': ticker['volume'],
                            'indicators': indicators,
                            'recent_candles': candles[-10:],
                        }
                        
                        # Constraints
                        constraints = {
                            'max_position_size': portfolio['max_position_size_usdt'],
                            'max_open_positions': settings.max_open_positions,
                            'min_confidence': settings.ai_min_confidence,
                            'risk_per_trade_pct': settings.risk_per_trade_pct,
                        }
                        
                        # Get AI decision
                        decision = await ai_strategy_service.analyze(
                            symbol=symbol,
                            market_data=market_data,
                            portfolio=portfolio,
                            constraints=constraints
                        )
                        
                        if not decision:
                            logger.warning(f"No decision from AI for {symbol}")
                            continue
                        
                        # Save signal
                        signal = AISignal(
                            symbol=symbol,
                            action=decision['action'],
                            confidence=decision['confidence'],
                            size_usdt=decision.get('size_usdt'),
                            reasoning=decision['reasoning'],
                            context=decision['context'],
                            stop_loss=decision.get('stop_loss'),
                            take_profit=decision.get('take_profit'),
                        )
                        db.add(signal)
                        db.commit()
                        
                        # Broadcast signal to connected clients
                        await manager.broadcast({
                            'type': 'ai_signal',
                            'timestamp': datetime.now().isoformat(),
                            'symbol': symbol,
                            'action': decision['action'],
                            'confidence': decision['confidence'],
                            'reasoning': decision['reasoning'],
                            'size_usdt': decision.get('size_usdt'),
                        })
                        
                        # Validate and execute if conditions met
                        if not ai_strategy_service.validate_decision(decision, portfolio):
                            logger.info(f"Decision validation failed for {symbol}")
                            continue
                        
                        # Execute based on action
                        action = decision['action']
                        
                        if action == 'BUY':
                            # Check if we can open position
                            can_open, reason = portfolio_service.can_open_position(
                                db, symbol, decision.get('size_usdt', 0)
                            )
                            
                            if not can_open:
                                logger.info(f"Cannot open position: {reason}")
                                continue
                            
                            # Calculate quantity
                            size_usdt = decision.get('size_usdt', 0)
                            quantity = size_usdt / current_price
                            
                            # Execute buy order
                            result = await execution_simulator.execute_market_order(
                                db=db,
                                exchange='binance',
                                symbol=symbol,
                                side='BUY',
                                quantity=quantity,
                                current_price=current_price,
                                ai_signal_id=signal.id
                            )
                            
                            if result:
                                signal.executed = True
                                db.commit()
                                
                                # Broadcast execution
                                await manager.broadcast({
                                    'type': 'order_filled',
                                    **result
                                })
                                
                                logger.info(f"Executed BUY: {result}")
                        
                        elif action == 'SELL' or action == 'CLOSE':
                            # Close existing position
                            result = await execution_simulator.close_position(
                                db=db,
                                symbol=symbol,
                                current_price=current_price,
                                ai_signal_id=signal.id
                            )
                            
                            if result:
                                signal.executed = True
                                db.commit()
                                
                                # Broadcast execution
                                await manager.broadcast({
                                    'type': 'order_filled',
                                    **result
                                })
                                
                                logger.info(f"Executed SELL/CLOSE: {result}")
                        
                        # Update portfolio and broadcast
                        updated_portfolio = portfolio_service.get_state(db, current_prices)
                        
                        # Save snapshot
                        portfolio_service.save_snapshot(db, updated_portfolio)
                        
                        # Broadcast portfolio update
                        await manager.broadcast({
                            'type': 'portfolio',
                            'timestamp': datetime.now().isoformat(),
                            **updated_portfolio
                        })
                
                except Exception as e:
                    logger.error(f"Error analyzing {symbol}: {e}", exc_info=True)
                    continue
            
            logger.info("AI analysis cycle completed")
            
        except asyncio.CancelledError:
            logger.info("AI trading loop cancelled")
            break
        except Exception as e:
            logger.error(f"Error in trading loop: {e}", exc_info=True)
            await asyncio.sleep(10)  # Wait before retry


@asynccontextmanager
async def lifespan(app: FastAPI):
    """
    Application lifespan manager - startup and shutdown
    """
    # Startup
    logger.info("Starting Crypto AI Trader...")
    
    # Initialize database tables
    Base.metadata.create_all(bind=engine)
    logger.info("Database tables initialized")
    
    # Initialize market data service
    await market_data_service.initialize()
    
    # Start market data streaming
    asyncio.create_task(
        market_data_service.start_streaming(settings.symbols_list)
    )
    
    # Subscribe to market data updates
    async def market_data_handler(data):
        """Handle market data updates"""
        await manager.broadcast(data)
    
    market_data_service.subscribe(market_data_handler)
    
    # Start AI trading loop
    global background_task
    background_task = asyncio.create_task(ai_trading_loop())
    
    logger.info("Application started successfully")
    
    yield
    
    # Shutdown
    logger.info("Shutting down...")
    
    # Cancel background task
    if background_task:
        background_task.cancel()
        try:
            await background_task
        except asyncio.CancelledError:
            pass
    
    # Stop market data streaming
    await market_data_service.stop_streaming()
    await market_data_service.close()
    
    logger.info("Application shut down successfully")


# Create FastAPI app
app = FastAPI(
    title="Crypto AI Trader",
    description="AI-powered cryptocurrency trading platform",
    version="1.0.0",
    lifespan=lifespan
)

# Configure CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],  # In production, specify exact origins
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Include routers
app.include_router(router)

# WebSocket route
@app.websocket("/ws")
async def websocket_route(websocket: WebSocket):
    await websocket_endpoint(websocket)


@app.get("/")
async def root():
    """Root endpoint"""
    return {
        "app": "Crypto AI Trader",
        "version": "1.0.0",
        "status": "running",
        "docs": "/docs"
    }


if __name__ == "__main__":
    import uvicorn
    uvicorn.run(
        "app.main:app",
        host=settings.backend_host,
        port=settings.backend_port,
        reload=True
    )
