"""
Market Data Service - CCXT integration for real-time market data
"""
import asyncio
import ccxt.pro as ccxtpro
from typing import Dict, List, Optional, Callable
from datetime import datetime
from app.config import settings
import logging

logger = logging.getLogger(__name__)


class MarketDataService:
    """
    Manages real-time market data streaming from multiple exchanges
    """
    
    def __init__(self):
        self.exchanges: Dict[str, ccxtpro.Exchange] = {}
        self.subscribers: List[Callable] = []
        self.running = False
        
    async def initialize(self):
        """Initialize exchange connections"""
        for exchange_id in settings.exchanges_list:
            try:
                exchange_class = getattr(ccxtpro, exchange_id)
                
                # Configure with API keys if available
                config = {
                    'enableRateLimit': True,
                }
                
                if exchange_id == 'binance' and settings.binance_api_key:
                    config['apiKey'] = settings.binance_api_key
                    config['secret'] = settings.binance_secret
                elif exchange_id == 'coinbase' and settings.coinbase_api_key:
                    config['apiKey'] = settings.coinbase_api_key
                    config['secret'] = settings.coinbase_secret
                elif exchange_id == 'kraken' and settings.kraken_api_key:
                    config['apiKey'] = settings.kraken_api_key
                    config['secret'] = settings.kraken_secret
                
                exchange = exchange_class(config)
                await exchange.load_markets()
                self.exchanges[exchange_id] = exchange
                logger.info(f"Initialized exchange: {exchange_id}")
                
            except Exception as e:
                logger.error(f"Failed to initialize {exchange_id}: {e}")
    
    def subscribe(self, callback: Callable):
        """Subscribe to market data updates"""
        self.subscribers.append(callback)
    
    async def broadcast(self, data: dict):
        """Broadcast data to all subscribers"""
        for callback in self.subscribers:
            try:
                await callback(data)
            except Exception as e:
                logger.error(f"Error broadcasting to subscriber: {e}")
    
    async def watch_ticker(self, exchange_id: str, symbol: str):
        """
        Watch ticker updates for a symbol on an exchange
        """
        exchange = self.exchanges.get(exchange_id)
        if not exchange:
            logger.error(f"Exchange {exchange_id} not initialized")
            return
        
        while self.running:
            try:
                ticker = await exchange.watch_ticker(symbol)
                
                data = {
                    'type': 'ticker',
                    'exchange': exchange_id,
                    'symbol': symbol,
                    'timestamp': ticker.get('timestamp', datetime.now().timestamp() * 1000),
                    'bid': ticker.get('bid'),
                    'ask': ticker.get('ask'),
                    'last': ticker.get('last'),
                    'volume': ticker.get('baseVolume'),
                    'high': ticker.get('high'),
                    'low': ticker.get('low'),
                }
                
                await self.broadcast(data)
                
            except Exception as e:
                logger.error(f"Error watching ticker {exchange_id} {symbol}: {e}")
                await asyncio.sleep(5)  # Wait before retry
    
    async def fetch_ohlcv(
        self, 
        exchange_id: str, 
        symbol: str, 
        timeframe: str = '5m', 
        limit: int = 100
    ) -> List[List]:
        """
        Fetch OHLCV candles for a symbol
        Returns: [[timestamp, open, high, low, close, volume], ...]
        """
        exchange = self.exchanges.get(exchange_id)
        if not exchange:
            raise ValueError(f"Exchange {exchange_id} not initialized")
        
        try:
            ohlcv = await exchange.fetch_ohlcv(symbol, timeframe, limit=limit)
            return ohlcv
        except Exception as e:
            logger.error(f"Error fetching OHLCV {exchange_id} {symbol}: {e}")
            return []
    
    async def fetch_ticker(self, exchange_id: str, symbol: str) -> Optional[dict]:
        """Fetch current ticker for a symbol"""
        exchange = self.exchanges.get(exchange_id)
        if not exchange:
            return None
        
        try:
            ticker = await exchange.fetch_ticker(symbol)
            return {
                'symbol': symbol,
                'exchange': exchange_id,
                'bid': ticker.get('bid'),
                'ask': ticker.get('ask'),
                'last': ticker.get('last'),
                'volume': ticker.get('baseVolume'),
                'timestamp': ticker.get('timestamp'),
            }
        except Exception as e:
            logger.error(f"Error fetching ticker {exchange_id} {symbol}: {e}")
            return None
    
    async def start_streaming(self, symbols: List[str]):
        """Start streaming market data for given symbols"""
        self.running = True
        tasks = []
        
        for exchange_id in self.exchanges.keys():
            for symbol in symbols:
                task = asyncio.create_task(
                    self.watch_ticker(exchange_id, symbol)
                )
                tasks.append(task)
        
        logger.info(f"Started streaming for {len(tasks)} symbol/exchange pairs")
        await asyncio.gather(*tasks, return_exceptions=True)
    
    async def stop_streaming(self):
        """Stop all streaming"""
        self.running = False
        logger.info("Stopping market data streaming")
    
    async def close(self):
        """Close all exchange connections"""
        for exchange_id, exchange in self.exchanges.items():
            try:
                await exchange.close()
                logger.info(f"Closed exchange: {exchange_id}")
            except Exception as e:
                logger.error(f"Error closing {exchange_id}: {e}")


# Global instance
market_data_service = MarketDataService()
